/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package rdc;


import com.inet.report.Element;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.FieldElement;
import com.inet.report.Fields;
import com.inet.report.FormulaField;
import com.inet.report.Section;
import com.inet.report.SpecialField;
import com.inet.report.ValueProperties;

/**
 * This example shows how to invoke a user-defined function. It iterates through all sections and gathers all elements
 * which should invoke the user-defined function. The function in this example is getDateFormat, which should be defined
 * for arguments java.sql.Date, java.sql.Timestamp and java.sql.Time in your user defined function class (set in your
 * configuration using Configuration Manager). Example for map function (getDateFormat here): String
 * getDateFormat(java.util.Date x){ return x.toString();//Or your own formatting routine. } public String
 * getDateFormat(java.sql.Date x){ return getDateFormat((java.util.Date)x); } public String getDateFormat(java.sql.Time
 * x){ return getDateFormat((java.util.Date)x); } public String getDateFormat(java.sql.Timestamp x){ return
 * getDateFormat((java.util.Date)x); }
 */
public class MapFunction extends RDCSample {

    //the relative path to the report file
    static final String MAINREPORT = "samples/rdc/mainrep.rpt";

    /**
     * Iterates through all sections and gathers all elements which should invoke the user-defined function.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            //create a new Engine
            Engine eng = new Engine( exportFmt );
            //set the name of the report file (as absolute path) (here relative to the users working directory)
            //			eng.setReportFile("file:" + System.getProperty("user.dir") + mainReport);
            eng.setReportFile( MAINREPORT );
            //eng.setPrompt("date","Time(1,25,00)");
            //eng.setPrompt("Date(1,25,00)",0);

            //to add formula fields, a fields set is needed.
            Fields fields = eng.getFields();
            //to find all elements, we start searching on all sections

            for( Section sec : eng.getSections() ) {
                for( Element elem : sec.getElements() ) {
                    //The element should be a field element
                    if( elem.getType() != Element.FIELD ) {
                        continue;
                    }

                    //The element doesn't have a customized format.
                    FieldElement felem = (FieldElement)elem;
                    if( felem.getDateFormatType() == ValueProperties.NOT_USING_SYSTEM_DEFAULTS ) {
                        continue;
                    }

                    //The element should have the value type Date/Time/Datetime
                    Field field = felem.getField();
                    if( field.getValueType() != Field.DATE && field.getValueType() != Field.DATETIME && field.getValueType() != Field.TIME ) {
                        continue;
                    }

                    //create a formula field which map the function to the field.
                    FormulaField formatted = null;
                    if( field.getType() != Field.SPECIAL_FIELD ) {
                        formatted =
                            fields.addFormulaField( "getDateFormat of " + field.getName(), "getDateFormat({" + field.getRefName() + "})",
                                                    FormulaField.FORMULA_USER );
                    } else {
                        //Special case are special fields, they can't be used in formulas, because for each special field 
                        //exists a function for formulas. 
                        SpecialField spField = (SpecialField)field;
                        formatted =
                            fields.addFormulaField( "getDateFormat of " + spField.getOperationName() + "()",
                                                    "getDateFormat(" + spField.getOperationName() + "())", FormulaField.FORMULA_USER );
                    }

                    //set the new formula, which map the function to the position of the field to map.
                    felem.setField( formatted );
                }
            }

            //return engine for execute
            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new MapFunction().initUI();
    }
}
